//  AUTHOR: blazr
//    DATE: OCT 2025
// VERSION: 1.8

// INITIALIZATION
echo( "---------------------------------" );
echo( "SCRIPT START" );
/* [BUILD/ADD] */
// CAP
USE_CAP = true;
// GRIP/HANDLE
USE_GRIP = true;

/* [CAP] */
// CAP RESOLUTION/QUALITY
cap_resolution = 300; // [20:Low,50:Medium,300:High,600:Perfect]
// COLOR
cap_color = "Silver"; // [Dimgray,Gray,Slategray,Silver,Lightgray,Gainsboro,Transparent]
// INNER DIAMETER (CM)
cap_diameter_inner = 20.7; // [10:0.1:100]
// HEIGHT (CM)
cap_height = 2.0;  // [1:0.5:50]
// WALL SIZE (CM)
cap_wall_size = 0.5; // [0.5:0.1:5]
// HOLES DIAMETER (CM)
hole_diameter = 0.55; // .01
// SCREW HEAD HEIGHT (CM)
hole_screw_head_length = 0.3; // .05
// SCREW HEAD DIAMETER (CM)
hole_screw_head_diameter = 1.05; // .01
// HEX NUT RADIUS (CM)
hole_hexa_radius = 0.5; // .05
// HEX NUT DEPTH (CM)
hole_hexa_depth = 0.3; // .1

/* [GRIP] */
// COLOR
grip_color = "Dimgray"; // [Dimgray,Gray,Slategray,Silver,Lightgray,Gainsboro,Transparent]
// HEIGHT (CM)
grip_height = 2.0; // [1:0.5:15]
// WIDTH (CM)
grip_width = 1.5; // [1:0.1:5]
// LENGTH (CM)
grip_length = 6.0; // [4:0.5:30]
// WALL SIZE (CM)
grip_wall_size = 0.5; // [0.5:0.1:5]

// USED TO STOP FOLLOWING VARS TO APPEAR IN CUSTOMIZER
module __Customizer_Limit__ () {}
 
// VERSION OF THE MODEL RELEASED
model_version = [1, 8, 0];
// VERSION OF THE OPENSCAD BINARY
openscad_version = version();
// TINY VALUE TO CORRECT SUBTRACTIONS
flicker_correction = 0.005; // NEEDED TO FIX OPENSCAD FLICKERING ON DIFFS IN PREVIEW-RENDER

/**
* ADJUST FINAL ORIENTATION OF OBJECT DEPENDING ON WHAT TO BUILD
*/
function all_rotation_x( value ) = USE_CAP ? ( USE_GRIP ? 0 : value ) : 0;

function grip_rotation_x( value ) = USE_CAP ? ( USE_GRIP ? 0 : 0 ) : value;

function grip_translation_y( value ) = USE_CAP ? ( USE_GRIP ? 0 : 0 ) : value;

/**
* ALLOWS USE OF ONE TRANSPARENT COLOR
*/
function recolor( color_name_original ) = color_name_original == "Transparent" ? [0.3,0.3,0.3,0.5] : color_name_original;

/**
* RENDERING PARAMETERS & VIEWPORT CONFIG
*/
module initialize( min_angle, min_size, num_fragments ) {
  // DEFINES THE MINIMUM ANGLE WE WILL BREAK IN ONE FRAGMENT
  $fa = min_angle;
  // DEFINES MINIMUM SIZE OF SOMETHINg TO BE RENDERED
  $fs = min_size;
  // DEFINES HOW MANY FRAGMENTS SHOULD BE USED TO RENDER
  $fn = num_fragments;
  echo( "--- VIEWPORT CONFIGURATION ---" );
  echo( str("ROTATION ANGLES: ",$vpr," DEGREES") );
  echo( str("TRANSLATIONS: ",$vpt," UNITS") );
  echo( str("CAM DISTANCE: ",$vpd," UNITS") );
  echo( str("FIELD OF VIEW: ",$vpf," UNITS") );
  echo( "------------------------------" );
}

/**
* CAP WITH A CERTAIN DEPTH AND WALL THICKNESS
*/
module cap( height, radius, wall_size, color_name ) {
  color( recolor(color_name) ) difference() {
    // OUTER CYLINDER
    color( recolor(color_name) )
    cylinder(h = height, r = radius, center = false, $fn=cap_resolution);
    // INNER CYLINDER
    height_inner = height - wall_size;
    translate([0.0, 0.0, -flicker_correction]) {
      color( recolor( color_name ) )
      cylinder(h = height_inner-flicker_correction, r = (radius-wall_size), center = false, $fn=cap_resolution);
    }
  }
}

/**
* POLYGON 2D-SHAPE FOR EXTRUSION
*/
module regular_polygon(order = 3, radius = 1) {
     angles=[ for (i = [0:order-1]) i*(360/order) ];
     coords=[ for (th=angles) [radius*cos(th), radius*sin(th)] ];
     polygon( coords );
 }

/**
* HOLE ROUND
*/
module hole( length, radius, color_name, offset_center_x ) {
   translate( [offset_center_x,0,0] ) {
     color(color_name) cylinder(length, radius, radius, $fn = 50);
   }
}

/**
* HOLE HEXAGON
*/
module hole_hexa(length, radius, color_name, offset_center_x, offset_z) {
  translate( [offset_center_x, 0, offset_z] ) {
    color( color_name ) linear_extrude( length ) regular_polygon( 6, radius );
  } 
}

/**
* MAIN MODULE TO CONSTRUCT CAP & GRIP
*/
module main(cap_diameter,cap_height,cap_wall_size,gh,gw,gs,gl) {
  cap_radius_inner = cap_diameter / 2.0;
  cap_radius_outer = (cap_diameter + (2*cap_wall_size)) / 2.0;
  // HOLE CALCULATIONS
  hole_radius = hole_diameter/2.0;
  hole_spacing = 0.5;      
  hole_length = (2*hole_spacing)+hole_diameter;
  bar_length = (2*hole_length) + gl;
  hole_grip_offset_center = (bar_length/2.0)-(hole_spacing+hole_radius);
  hole_through_length = gs;
  // EXECUTE SOME VALIDATION CHECKS
  assert( cap_diameter > bar_length, "Grip length too long for cap." );
  assert( cap_wall_size < cap_height, "Cap wallsize exceeds cap height." );
  assert( cap_wall_size < (cap_diameter/2), "Cap wallsize seems to be way too large for this cap." );
  assert( gs*2 < gh, "Grip wallsize seems too big." );
  
  rotate([all_rotation_x( 180.0 ), 0.0, 0.0]) { // CONSTRUCTION ALL
    if( USE_CAP ) {
    translate([0,0,-cap_height]) { // CAP CONSTRUTION
      cap_radius = cap_radius_outer;
      %difference() { // HOLES HEXAGON
        // DECKEL
        difference() { // HOLES ROUND
          // CAP (BUILD WITH INNER & OUTER CYLINDER)
          cap(
            cap_height,
            cap_radius,
            cap_wall_size,
            cap_color
          );
          
          // HOLES ROUND THROUGH
          hole_offset_center = hole_grip_offset_center;
          hole_length = cap_height+flicker_correction;
          hole_diameter = 0.5;          
          hole_radius = hole_diameter/2.0;
          // HOLE 1
          hole( hole_length, hole_radius, "yellow", -hole_offset_center );
          // HOLE 2
          hole( hole_length, hole_radius, "yellow", hole_offset_center );
          
      } // difference HOLES ROUND
    
      // HOLES HEXAGON SURFACE
      hole_hexa_offset_center = hole_grip_offset_center;
      hole_hexa_offset_z = cap_height-cap_wall_size-(3*flicker_correction);
      hole_hexa_length = hole_hexa_depth;
      // HOLE HEXA 1
      hole_hexa(
        hole_hexa_length,
        hole_hexa_radius,
        "blue",
        hole_hexa_offset_center,
        hole_hexa_offset_z
      );
      // HOLE HEXA 2
      hole_hexa( 
        hole_hexa_length,
        hole_hexa_radius,
        "blue",
        -hole_hexa_offset_center,
        hole_hexa_offset_z
      );
    } // difference HOLES HEXAGON
  } // translate CAP CONSTRUTION
  } // if
  // END OF CAP CONSTRUCTION
  
  // BEGIN GRIP CONSTRUCTION (DEACTIVATED)
  if( USE_GRIP ) {
  rotate( [grip_rotation_x( 90 ),0,0] ) {
    translate( [0,grip_translation_y( +gw/2 ),0] ) {
  difference() {
    translate( [0,0,(gh/2.0)] ) {
      // HANDLE UNION
      union() {
        // UPPER PART WITH HOLE
        difference() {
            color( recolor( grip_color ) )
            cube([gl, gw, gh], center = true);
            translate( [0,0,-gs] ) {
              color(grip_color)
              cube([gl-(gs*2), (2.0*gw), (gh)], center = true);
            }
        }
        // LOWER PART
        translate( [0,0,-((gh-gs)/2.0)] ) {
          color( recolor( grip_color ) )
          cube([bar_length, gw, gs], center = true);
        }
      }
    }
    // HOLES
    translate( [-hole_grip_offset_center,0,-flicker_correction] ) {
      color("yellow") cylinder(hole_through_length+(2*flicker_correction),hole_radius,hole_radius, $fn = 50);
    }
    translate( [hole_grip_offset_center,0,-flicker_correction] ) {
      color("yellow") cylinder(hole_through_length+(2*flicker_correction),hole_radius,hole_radius, $fn = 50);
    }
    // HOLES HEAD
    hole_screw_head_radius = hole_screw_head_diameter/2.0;        
    translate( [-hole_grip_offset_center,0,(hole_through_length-hole_screw_head_length+flicker_correction)] ) {
      color("crimson") cylinder(hole_screw_head_length,hole_screw_head_radius,hole_screw_head_radius, $fn = 100);
    }
    translate( [hole_grip_offset_center,0,(hole_through_length-hole_screw_head_length+flicker_correction)] ) {
      color("crimson") cylinder(hole_screw_head_length,hole_screw_head_radius,hole_screw_head_radius, $fn = 100);
    }        
  } // difference
  } // rotate
  } // translate
  } // if USE
  } // rotation CONSTRUCTION ALL
} // END MAIN MODULE

// INIT RENDER SYSTEM
echo( "INITIALIZING" );
initialize(
  0.4,
  0.01,
  50
);

// EXECUTE PROGRAM
echo( "EXECUTING" );
main( 
  cap_diameter_inner,
  cap_height,
  cap_wall_size,
  grip_height,
  grip_width,
  grip_wall_size,
  grip_length
);
echo( "COMPLETE" );

echo( str( "MODEL VERSION: ", model_version[0],".",model_version[1],".",model_version[2] ) )
echo( str( "BINARY VERSION: ", openscad_version[0],".",openscad_version[1],".",openscad_version[2]) );
echo( "SCRIPT END" );
echo( "---------------------------------" );

/*
RELEASE HISTORY
1.7.0
* improved parameters for grip/handlebar

1.8.0
* added support for transparent colors
* added assert statements for boundary checks
* added init of rendersystem
* added output of viewport configuration
*/