//  AUTHOR: blazr
//    DATE: OCT 2025
// VERSION: 1.5

// INITIALIZATION
echo( "---------------------------------" );
echo( "INITIALIZING" );
/* [BUILD/ADD] */
// CAP
USE_CAP = true;
// GRIP/HANDLE
USE_GRIP = true;
// LOGO
USE_LOGO = false;
// LABELS
USE_LABELS = false;

/* [CAP] */
// CAP RESOLUTION/QUALITY
cap_resolution = 300; // [20:Low,50:Medium,300:High,600:Perfect]
// COLOR
cap_color = "Silver"; // [Dimgray,Gray,Slategray,Silver,Lightgray,Gainsboro]
// INNER DIAMETER (CM)
cap_diameter_inner = 20.7; // [10:0.1:100]
// HEIGHT (CM)
cap_height = 2.0;  // [1:0.5:50]
// WALL SIZE (CM)
cap_wall_size = 0.5; // [0.5:0.1:5]
// HOLES DIAMETER (CM)
hole_diameter = 0.55; // .01
// SCREW HEAD HEIGHT (CM)
hole_screw_head_length = 0.3; // .05
// SCREW HEAD DIAMETER (CM)
hole_screw_head_diameter = 1.05; // .01
// HEX NUT RADIUS (CM)
hole_hexa_radius = 0.5; // .05
// HEX NUT DEPTH (CM)
hole_hexa_depth = 0.3; // .1

/* [GRIP] */
// COLOR
grip_color = "Dimgray"; // [Dimgray,Gray,Slategray,Silver,Lightgray,Gainsboro]
// HEIGHT (CM)
grip_height = 2.0; // [1:0.5:15]
// WIDTH (CM)
grip_width = 1.5; // [1:0.1:5]
// LENGTH (CM)
grip_length = 6.0; // [4:0.5:30]
// WALL SIZE (CM)
grip_wall_size = 0.5; // [0.5:0.1:5]

/* [LOGO] */
// COLOR
logo_color = "Fuchsia"; // [Fuchsia,DeepPink,Crimson,DarkOrange,Gold,Yellow,GreenYellow,DarkViolet,Aqua,SandyBrown,Wheat]
// SIZE
logo_scale_size = 0.02;  // [0.0025:Tiny,0.005:Small,0.01:Normal,0.02:Bigger,0.04:Big]
// FILENAME (SVG)
logo_filename = "lightbulb.svg";

/* [LABEL] */
// COLOR
label_color = "Gold"; // [Fuchsia,DeepPink,Crimson,DarkOrange,Gold,Yellow,GreenYellow,DarkViolet,Aqua,SandyBrown,Wheat]
// TEXT
label_text = "blazr";
// FONT NAME
label_fontname = "Data Control";
// FONT HEIGHT (CM)
label_size = 3; // [0.1:0.1:10]

// USED TO STOP FOLLOWINg VARS TO APPEAR In CUSTOMIZER
module __Customizer_Limit__ () {}
 
// MISC VARS
model_version = [1, 5, 0];
openscad_version = version();
flicker_correction = 0.005; // NEEDED TO FIX OPENSCAD FLICKERING ON DIFFS IN PREVIEW-RENDER

/**
* CAP WITH A CERTAIN DEPTH AND WALL THICKNESS
*/
module cap( height, radius, wall_size, color_name, color_name_inner ) {
  difference() {
    // OUTER CYLINDER
    color( color_name )
    cylinder(h = height, r = radius, center = false, $fn=cap_resolution);
    // INNER CYLINDER
    height_inner = height - wall_size;
    translate([0.0, 0.0, -flicker_correction]) {
      color( color_name_inner )
      cylinder(h = height_inner-flicker_correction, r = (radius-wall_size), center = false, $fn=cap_resolution);
    }
  }
}

/**
* POLYGON 2D-SHAPE FOR EXTRUSION
*/
module regular_polygon(order = 3, radius = 1) {
     angles=[ for (i = [0:order-1]) i*(360/order) ];
     coords=[ for (th=angles) [radius*cos(th), radius*sin(th)] ];
     polygon( coords );
 }

/**
* HOLE ROUND
*/
module hole( length, radius, color_name, offset_center_x ) {
   translate( [offset_center_x,0,0] ) {
     color(color_name) cylinder(length, radius, radius, $fn = 50);
   }
}

/**
* HOLE HEXAGON
*/
module hole_hexa(length, radius, color_name, offset_center_x, offset_z) {
  translate( [offset_center_x, 0, offset_z] ) {
    color( color_name ) linear_extrude( length ) regular_polygon( 6, radius );
  } 
}

/**
* MAIN MODULE TO CONSTRUCT CAP & GRIP
*/
module main(cap_diameter,cap_height,cap_wall_size,label_text,gh,gw,gs,gl) {
  cap_radius_inner = cap_diameter / 2.0;
  cap_radius_outer = (cap_diameter + (2*cap_wall_size)) / 2.0;
  // HOLE CALCULATIONS
  hole_radius = hole_diameter/2.0;
  hole_spacing = 0.5;      
  hole_length = (2*hole_spacing)+hole_diameter;
  bar_length = (2*hole_length) + gl;
  hole_grip_offset_center = (bar_length/2.0)-(hole_spacing+hole_radius);
  hole_through_length = gs;  
  rotate([0.0, 180.0, 0.0]) { // CONSTRUCTION ALL
    if( USE_CAP ) {
    translate([0,0,-cap_height]) { // CAP CONSTRUTION
      cap_radius = cap_radius_outer;
      difference() { // HOLES HEXAGON
        // DECKEL
        difference() { // HOLES ROUND
          // CAP (BUILD WITH INNER & OUTER CYLINDER)
          cap(
            cap_height,
            cap_radius,
            cap_wall_size,
            cap_color,
            "gray"
          );
          
          // HOLES ROUND THROUGH
          hole_offset_center = hole_grip_offset_center;
          hole_length = cap_height+flicker_correction;
          hole_diameter = 0.5;          
          hole_radius = hole_diameter/2.0;
          // HOLE 1
          hole( hole_length, hole_radius, "yellow", -hole_offset_center );
          // HOLE 2
          hole( hole_length, hole_radius, "yellow", hole_offset_center );
          
      } // difference HOLES ROUND
    
      // HOLES HEXAGON SURFACE
      hole_hexa_offset_center = hole_grip_offset_center;
      hole_hexa_offset_z = cap_height-cap_wall_size-(3*flicker_correction);
      hole_hexa_length = hole_hexa_depth;
      // HOLE HEXA 1
      hole_hexa(
        hole_hexa_length,
        hole_hexa_radius,
        "blue",
        hole_hexa_offset_center,
        hole_hexa_offset_z
      );
      // HOLE HEXA 2
      hole_hexa( 
        hole_hexa_length,
        hole_hexa_radius,
        "blue",
        -hole_hexa_offset_center,
        hole_hexa_offset_z
      );
    } // difference HOLES HEXAGON
        
    // OPTIONAL LABELS ON TOP OF CAP (PROBLEMATIC FOR PRINTING)
    if( USE_LABELS ) {
      label_height = 0.1;
      label_offset_to_center = cap_radius_outer/2.3;
      // LABEL LOWER
      translate([0.0, +(label_offset_to_center-(label_size/2.0)), cap_height]) {
        color(label_color) linear_extrude(label_height) text(label_text, size=label_size,halign="center",font=label_fontname, $fn=300);
      }
      
      // LABEL UPPER
      rotate( [0, 0, 180] ) {
        translate( [0.0, +(label_offset_to_center-(label_size/2.0)), cap_height] ) {
          color(label_color) linear_extrude(label_height) text(label_text, size=label_size,halign="center",font=label_fontname, $fn=300);
        }
      }
    } // if
  } // translate CAP CONSTRUTION
  } // if
  // END OF CAP CONSTRUCTION
  
  // BEGIN GRIP CONSTRUCTION (DEACTIVATED)
  if( USE_GRIP ) {
  difference() {
    translate( [0,0,(gh/2.0)] ) {
      // HANDLE UNION
      union() {
        // UPPER PART WITH HOLE
        difference() {
            color(grip_color)
            cube([gl, gw, gh], center = true);
            translate( [0,0,-gs] ) {
              color(grip_color)
              cube([gl-(gs*2), (2.0*gw), (gh)], center = true);
            }
        }
        // LOWER PART
        translate( [0,0,-((gh-gs)/2.0)] ) {
          color(grip_color)
          cube([bar_length, gw, gs], center = true);
        }
      }
    }
    // HOLES
    translate( [-hole_grip_offset_center,0,-flicker_correction] ) {
      color("yellow") cylinder(hole_through_length+(2*flicker_correction),hole_radius,hole_radius, $fn = 50);
    }
    translate( [hole_grip_offset_center,0,-flicker_correction] ) {
      color("yellow") cylinder(hole_through_length+(2*flicker_correction),hole_radius,hole_radius, $fn = 50);
    }
    // HOLES HEAD
    hole_screw_head_radius = hole_screw_head_diameter/2.0;        
    translate( [-hole_grip_offset_center,0,(hole_through_length-hole_screw_head_length+flicker_correction)] ) {
      color("crimson") cylinder(hole_screw_head_length,hole_screw_head_radius,hole_screw_head_radius, $fn = 100);
    }
    translate( [hole_grip_offset_center,0,(hole_through_length-hole_screw_head_length+flicker_correction)] ) {
      color("crimson") cylinder(hole_screw_head_length,hole_screw_head_radius,hole_screw_head_radius, $fn = 100);
    }        
  } // difference
  }

  // OPTIONAL LOGO
  if( USE_LOGO ) {
    // ADDITIONAL STUFF
    color(logo_color) linear_extrude(0.1)
    translate( [-7.3,0, 0] ) {
      scale([logo_scale_size,logo_scale_size,logo_scale_size]) {
          import(file = logo_filename, center = true, $fn = 100);
        }
      }
      mirror( [0,1,0] ) {
        color(logo_color) linear_extrude(0.1)
        translate( [7.3,0, 0] ) {
          scale([logo_scale_size,logo_scale_size,logo_scale_size]) {
            import(file = logo_filename, center = true, $fn = 100);
          }
        }
      }
    }    
  } // rotation CONSTRUCTION ALL
} // END MAIN MODULE

// EXECUTE PROGRAM
echo( "EXECUTING" );
main( 
  cap_diameter_inner,
  cap_height,
  cap_wall_size,
  label_text,
  grip_height,
  grip_width,
  grip_wall_size,
  grip_length
);

echo( str( "MODEL VERSION: ", model_version[0],".",model_version[1],".",model_version[2] ) )
echo( str( "BINARY VERSION: ", openscad_version[0],".",openscad_version[1],".",openscad_version[2]) );
echo( "COMPLETE" );
echo( "---------------------------------" );